<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\ProductController;
use App\Http\Controllers\Api\StoreController;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\OrderController;
use App\Http\Controllers\Api\CategoryController;
use App\Http\Controllers\Api\StockController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\SettingsController;
use App\Http\Controllers\Api\PromotionController;
use App\Http\Controllers\Api\DashboardController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// API v1 routes
Route::prefix('v1')->group(function () {
    // Public routes
    Route::get('/products', [ProductController::class, 'index']);
    Route::get('/products/{id}', [ProductController::class, 'show']);
    Route::get('/stores', [StoreController::class, 'index']);
    Route::get('/stores/{id}', [StoreController::class, 'show']);
    Route::get('/categories', [CategoryController::class, 'index']);
    Route::get('/categories/{id}', [CategoryController::class, 'show']);

    // Auth routes
    Route::post('/auth/register', [AuthController::class, 'register']);
    Route::post('/auth/verify-email', [AuthController::class, 'verifyEmail']);
    Route::post('/auth/resend-code', [AuthController::class, 'resendVerificationCode']);
    Route::post('/auth/login', [AuthController::class, 'login']);

    // Order routes (public)
    Route::post('/orders', [OrderController::class, 'store']);
    Route::get('/orders/{uuid}', [OrderController::class, 'show']); // UUID pour sécurité

    // Active promotions (public for logged-in clients)
    Route::get('/promotions/active', [PromotionController::class, 'active']);

    // Protected routes (authentification requise)
    Route::middleware('auth:sanctum')->group(function () {
        Route::post('/auth/logout', [AuthController::class, 'logout']);
        Route::get('/auth/me', [AuthController::class, 'me']);
    });

    // Admin routes (auth + role ADMIN + is_active)
    Route::middleware(['auth:sanctum', 'admin'])->group(function () {
        // Products management (admin only)
        Route::post('/products', [ProductController::class, 'store']);
        Route::put('/products/{id}', [ProductController::class, 'update']);
        Route::delete('/products/{id}', [ProductController::class, 'destroy']);
        Route::patch('/products/{id}/toggle-active', [ProductController::class, 'toggleActive']);
        Route::post('/products/upload-images', [ProductController::class, 'uploadImages']);
        Route::post('/products/delete-image', [ProductController::class, 'deleteImage']);

        // Stores management (admin only)
        Route::post('/stores', [StoreController::class, 'store']);
        Route::put('/stores/{id}', [StoreController::class, 'update']);
        Route::delete('/stores/{id}', [StoreController::class, 'destroy']);
        Route::patch('/stores/{id}/toggle-active', [StoreController::class, 'toggleActive']);

        // Categories management (admin only)
        Route::post('/categories', [CategoryController::class, 'store']);
        Route::put('/categories/{id}', [CategoryController::class, 'update']);
        Route::delete('/categories/{id}', [CategoryController::class, 'destroy']);

        // Stock management (admin only)
        Route::get('/stock/global', [StockController::class, 'getGlobalStats']);
        Route::get('/stock/alerts', [StockController::class, 'getAlerts']);
        Route::get('/stock/store/{storeId}', [StockController::class, 'getStoreStock']);
        Route::post('/stock/adjust', [StockController::class, 'adjustStock']);

        // Orders management (admin only)
        Route::get('/orders', [OrderController::class, 'index']);
        Route::patch('/orders/{id}/status', [OrderController::class, 'updateStatus']);

        // Users management (admin only)
        Route::get('/users', [UserController::class, 'index']);
        Route::get('/users/{id}', [UserController::class, 'show']);
        Route::post('/users', [UserController::class, 'store']);
        Route::put('/users/{id}', [UserController::class, 'update']);
        Route::patch('/users/{id}/toggle-active', [UserController::class, 'toggleActive']);
        Route::patch('/users/{id}/activate', [UserController::class, 'activate']);
        Route::delete('/users/{id}', [UserController::class, 'destroy']);

        // Settings management (admin only)
        Route::get('/settings', [SettingsController::class, 'index']);
        Route::get('/settings/{key}', [SettingsController::class, 'show']);
        Route::put('/settings/{key}', [SettingsController::class, 'update']);
        Route::post('/settings/batch', [SettingsController::class, 'updateBatch']);
        Route::delete('/settings/{key}', [SettingsController::class, 'destroy']);

        // Promotions management (admin only)
        Route::get('/promotions', [PromotionController::class, 'index']);
        Route::get('/promotions/{id}', [PromotionController::class, 'show']);
        Route::post('/promotions', [PromotionController::class, 'store']);
        Route::put('/promotions/{id}', [PromotionController::class, 'update']);
        Route::patch('/promotions/{id}/toggle-active', [PromotionController::class, 'toggleActive']);
        Route::delete('/promotions/{id}', [PromotionController::class, 'destroy']);

        // Dashboard statistics (admin only)
        Route::get('/dashboard/stats', [DashboardController::class, 'stats']);
    });
});

Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});
