<?php

namespace App\Services;

use App\Models\StockItem;
use App\Models\ProductVariant;
use App\Models\Setting;
use App\Mail\LowStockAlert;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\Mail;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\StockReservationException;

class StockService
{
    /**
     * Durée de réservation temporaire en secondes (15 min par défaut)
     */
    protected int $reservationTTL;

    public function __construct()
    {
        $this->reservationTTL = config('app.stock_reservation_ttl', 900); // 15 min
    }

    /**
     * Vérifier la disponibilité du stock dans un magasin
     *
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @return bool
     */
    public function checkAvailability(int $productId, int $variantId, int $storeId, int $quantity = 1): bool
    {
        $stockItem = StockItem::where('product_id', $productId)
            ->where('variant_id', $variantId)
            ->where('store_id', $storeId)
            ->first();

        if (!$stockItem) {
            return false;
        }

        // Utiliser available_quantity au lieu de quantity
        return $stockItem->available_quantity >= $quantity;
    }

    /**
     * Réserver temporairement du stock (15 min)
     * Utilise Redis pour la réservation temporaire
     *
     * @param string $cartId
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @return bool
     * @throws StockReservationException
     */
    public function reserveStock(string $cartId, int $productId, int $variantId, int $storeId, int $quantity): bool
    {
        try {
            $reservationKey = $this->getReservationKey($cartId, $productId, $variantId);

            // Vérifier disponibilité
            if (!$this->checkAvailability($productId, $variantId, $storeId, $quantity)) {
                throw new InsufficientStockException("Stock insuffisant pour ce produit dans ce magasin");
            }

            // Créer réservation temporaire dans Redis
            $reservationData = [
                'cart_id' => $cartId,
                'product_id' => $productId,
                'variant_id' => $variantId,
                'store_id' => $storeId,
                'quantity' => $quantity,
                'reserved_at' => now()->toDateTimeString(),
            ];

            Redis::setex(
                $reservationKey,
                $this->reservationTTL,
                json_encode($reservationData)
            );

            return true;
        } catch (\Exception $e) {
            throw new StockReservationException("Erreur lors de la réservation: {$e->getMessage()}");
        }
    }

    /**
     * Libérer une réservation temporaire
     *
     * @param string $cartId
     * @param int $productId
     * @param int $variantId
     * @return bool
     */
    public function releaseReservation(string $cartId, int $productId, int $variantId): bool
    {
        $reservationKey = $this->getReservationKey($cartId, $productId, $variantId);
        return Redis::del($reservationKey) > 0;
    }

    /**
     * Réserver du stock pour une commande (increment reserved_quantity)
     * Utilisé lors de la création de commande
     *
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @param string|null $reason
     * @param int|null $performedBy
     * @return StockItem
     * @throws InsufficientStockException
     */
    public function reserveStockForOrder(
        int $productId,
        int $variantId,
        int $storeId,
        int $quantity,
        ?string $reason = null,
        ?int $performedBy = null
    ): StockItem {
        return DB::transaction(function () use ($productId, $variantId, $storeId, $quantity, $reason, $performedBy) {
            // SELECT ... FOR UPDATE (pessimistic lock)
            $stockItem = StockItem::where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->where('store_id', $storeId)
                ->lockForUpdate()
                ->first();

            if (!$stockItem) {
                throw new InsufficientStockException("Stock non trouvé pour ce produit dans ce magasin");
            }

            // Vérification critique : Stock disponible (quantity - reserved_quantity)
            $availableStock = $stockItem->quantity - $stockItem->reserved_quantity;

            if ($availableStock < $quantity) {
                throw new InsufficientStockException(
                    "Stock disponible insuffisant: demandé {$quantity}, disponible {$availableStock} " .
                    "(physique: {$stockItem->quantity}, réservé: {$stockItem->reserved_quantity})"
                );
            }

            // Incrémenter reserved_quantity (quantity reste inchangé)
            $stockItem->reserved_quantity += $quantity;
            $stockItem->save();

            // Log du mouvement
            $stockItem->logMovement(
                'RESERVE',
                $quantity,
                $reason ?? 'Réservation - nouvelle commande',
                $performedBy
            );

            return $stockItem;
        });
    }

    /**
     * Libérer du stock réservé (decrement reserved_quantity)
     * Utilisé lors de l'annulation de commande
     *
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @param string|null $reason
     * @param int|null $performedBy
     * @return StockItem
     */
    public function releaseStockForOrder(
        int $productId,
        int $variantId,
        int $storeId,
        int $quantity,
        ?string $reason = null,
        ?int $performedBy = null
    ): StockItem {
        return DB::transaction(function () use ($productId, $variantId, $storeId, $quantity, $reason, $performedBy) {
            $stockItem = StockItem::where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->where('store_id', $storeId)
                ->lockForUpdate()
                ->first();

            if (!$stockItem) {
                throw new \Exception("Stock non trouvé pour ce produit dans ce magasin");
            }

            // Décrémenter reserved_quantity (quantity reste inchangé)
            // Protection contre valeurs négatives
            $stockItem->reserved_quantity = max(0, $stockItem->reserved_quantity - $quantity);
            $stockItem->save();

            // Log du mouvement
            $stockItem->logMovement(
                'RELEASE',
                -$quantity,
                $reason ?? 'Libération - annulation commande',
                $performedBy
            );

            return $stockItem;
        });
    }

    /**
     * Confirmer la sortie du stock physique lors de la livraison
     * Décrémente quantity ET reserved_quantity
     *
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @param string|null $reason
     * @param int|null $performedBy
     * @return StockItem
     * @throws InsufficientStockException
     */
    public function confirmStockForDelivery(
        int $productId,
        int $variantId,
        int $storeId,
        int $quantity,
        ?string $reason = null,
        ?int $performedBy = null
    ): StockItem {
        return DB::transaction(function () use ($productId, $variantId, $storeId, $quantity, $reason, $performedBy) {
            $stockItem = StockItem::where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->where('store_id', $storeId)
                ->lockForUpdate()
                ->first();

            if (!$stockItem) {
                throw new InsufficientStockException("Stock non trouvé pour ce produit dans ce magasin");
            }

            // Vérifier que le stock physique est suffisant
            if ($stockItem->quantity < $quantity) {
                throw new InsufficientStockException(
                    "Stock physique insuffisant: demandé {$quantity}, disponible {$stockItem->quantity}"
                );
            }

            // Décrémenter quantity (sortie physique)
            $stockItem->quantity -= $quantity;

            // Décrémenter reserved_quantity (libération réservation)
            $stockItem->reserved_quantity = max(0, $stockItem->reserved_quantity - $quantity);

            $stockItem->save();

            // Log du mouvement
            $stockItem->logMovement(
                'OUT',
                -$quantity,
                $reason ?? 'Livraison - commande client',
                $performedBy
            );

            // Vérifier si le stock est critique
            $this->checkLowStockAlert($stockItem);

            return $stockItem;
        });
    }

    /**
     * Décrémenter le stock (avec verrouillage pessimiste)
     *
     * ⚠️ DÉPRÉCIÉ pour les commandes : Utiliser reserveStockForOrder() + confirmStockForDelivery()
     * Conservé uniquement pour ajustements manuels administratifs
     *
     * @deprecated Utiliser reserveStockForOrder() pour commandes
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @param int|null $performedBy
     * @param string|null $reason
     * @return StockItem
     * @throws InsufficientStockException
     */
    public function decrementStock(
        int $productId,
        int $variantId,
        int $storeId,
        int $quantity,
        ?int $performedBy = null,
        ?string $reason = null
    ): StockItem {
        return DB::transaction(function () use ($productId, $variantId, $storeId, $quantity, $performedBy) {
            // SELECT ... FOR UPDATE (pessimistic lock)
            $stockItem = StockItem::where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->where('store_id', $storeId)
                ->lockForUpdate()
                ->first();

            if (!$stockItem) {
                throw new InsufficientStockException("Stock non trouvé pour ce produit dans ce magasin");
            }

            if ($stockItem->quantity < $quantity) {
                throw new InsufficientStockException(
                    "Stock insuffisant: demandé {$quantity}, disponible {$stockItem->quantity}"
                );
            }

            // Décrémentation
            $stockItem->quantity -= $quantity;
            $stockItem->save();

            // Log du mouvement
            $stockItem->logMovement(
                'OUT',
                -$quantity,
                $reason ?? 'Vente - commande client',
                $performedBy
            );

            // Vérifier si le stock est critique
            $this->checkLowStockAlert($stockItem);

            return $stockItem;
        });
    }

    /**
     * Vérifier et envoyer une alerte si le stock est critique
     */
    protected function checkLowStockAlert(StockItem $stockItem): void
    {
        // Vérifier si les alertes sont activées
        if (!Setting::get('stock_alert_enabled', true)) {
            return;
        }

        $threshold = Setting::get('stock_alert_threshold', 5);

        // Si le stock est en dessous du seuil
        if ($stockItem->available_quantity <= $threshold) {
            $this->sendLowStockAlert($stockItem);
        }
    }

    /**
     * Envoyer une alerte de stock critique
     */
    protected function sendLowStockAlert(StockItem $stockItem): void
    {
        // Charger les relations pour l'email
        $stockItem->load(['product', 'variant', 'store']);

        // Récupérer les emails des admins
        $adminEmails = Setting::get('admin_notification_emails', []);

        if (empty($adminEmails)) {
            return;
        }

        // Dispatch en queue (non bloquant pour l'operation)
        try {
            foreach ($adminEmails as $email) {
                Mail::to($email)->queue(new LowStockAlert([$stockItem]));
            }
        } catch (\Exception $e) {
            // Log l'erreur mais ne pas bloquer l'opération
            logger()->error('Failed to dispatch low stock alert email', [
                'stock_item_id' => $stockItem->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Incrémenter le stock (réapprovisionnement)
     *
     * @param int $productId
     * @param int $variantId
     * @param int $storeId
     * @param int $quantity
     * @param string|null $reason
     * @param int|null $performedBy
     * @return StockItem
     */
    public function incrementStock(
        int $productId,
        int $variantId,
        int $storeId,
        int $quantity,
        ?string $reason = null,
        ?int $performedBy = null
    ): StockItem {
        return DB::transaction(function () use ($productId, $variantId, $storeId, $quantity, $reason, $performedBy) {
            $stockItem = StockItem::where('product_id', $productId)
                ->where('variant_id', $variantId)
                ->where('store_id', $storeId)
                ->lockForUpdate()
                ->first();

            if (!$stockItem) {
                // Créer le stock s'il n'existe pas
                $stockItem = StockItem::create([
                    'product_id' => $productId,
                    'variant_id' => $variantId,
                    'store_id' => $storeId,
                    'quantity' => $quantity,
                ]);
            } else {
                $stockItem->quantity += $quantity;
                $stockItem->save();
            }

            // Log du mouvement
            $stockItem->logMovement(
                'IN',
                $quantity,
                $reason ?? 'Réapprovisionnement',
                $performedBy
            );

            return $stockItem;
        });
    }

    /**
     * Ajuster le stock manuellement (correction d'inventaire)
     *
     * @param int $stockItemId
     * @param int $newQuantity
     * @param string|null $reason
     * @param int|null $performedBy
     * @return StockItem
     */
    public function adjustStock(
        int $stockItemId,
        int $newQuantity,
        ?string $reason = null,
        ?int $performedBy = null
    ): StockItem {
        return DB::transaction(function () use ($stockItemId, $newQuantity, $reason, $performedBy) {
            $stockItem = StockItem::lockForUpdate()->findOrFail($stockItemId);

            $oldQuantity = $stockItem->quantity;
            $difference = $newQuantity - $oldQuantity;

            $stockItem->quantity = $newQuantity;
            $stockItem->save();

            // Log du mouvement
            $stockItem->logMovement(
                'ADJUSTMENT',
                $difference,
                $reason ?? 'Ajustement manuel',
                $performedBy
            );

            return $stockItem;
        });
    }

    /**
     * Transférer du stock entre magasins
     *
     * @param int $productId
     * @param int $variantId
     * @param int $fromStoreId
     * @param int $toStoreId
     * @param int $quantity
     * @param int|null $performedBy
     * @return array
     * @throws InsufficientStockException
     */
    public function transferStock(
        int $productId,
        int $variantId,
        int $fromStoreId,
        int $toStoreId,
        int $quantity,
        ?int $performedBy = null
    ): array {
        return DB::transaction(function () use ($productId, $variantId, $fromStoreId, $toStoreId, $quantity, $performedBy) {
            // Décrémenter stock source
            $fromStock = $this->decrementStock($productId, $variantId, $fromStoreId, $quantity, $performedBy);

            // Incrémenter stock destination
            $toStock = $this->incrementStock(
                $productId,
                $variantId,
                $toStoreId,
                $quantity,
                "Transfert depuis magasin #{$fromStoreId}",
                $performedBy
            );

            return [
                'from' => $fromStock,
                'to' => $toStock,
            ];
        });
    }

    /**
     * Obtenir les magasins avec stock disponible pour un produit
     *
     * @param int $productId
     * @param int $variantId
     * @param int $minQuantity
     * @return \Illuminate\Support\Collection
     */
    public function getStoresWithStock(int $productId, int $variantId, int $minQuantity = 1)
    {
        return StockItem::with('store')
            ->where('product_id', $productId)
            ->where('variant_id', $variantId)
            ->where('quantity', '>=', $minQuantity)
            ->get()
            ->pluck('store');
    }

    /**
     * Obtenir les items en stock faible pour un magasin
     *
     * @param int $storeId
     * @return \Illuminate\Support\Collection
     */
    public function getLowStockItems(int $storeId)
    {
        return StockItem::with(['product', 'variant'])
            ->where('store_id', $storeId)
            ->lowStock()
            ->get();
    }

    /**
     * Obtenir le stock theorique pour un magasin (stock physique actuel)
     *
     * @param int $storeId
     * @return \Illuminate\Support\Collection
     */
    public function getTheoreticalStockForStore(int $storeId)
    {
        return StockItem::with(['product', 'variant'])
            ->where('store_id', $storeId)
            ->orderBy('product_id')
            ->orderBy('variant_id')
            ->get();
    }

    /**
     * Générer la clé Redis pour une réservation
     *
     * @param string $cartId
     * @param int $productId
     * @param int $variantId
     * @return string
     */
    protected function getReservationKey(string $cartId, int $productId, int $variantId): string
    {
        return "stock:reservation:{$cartId}:{$productId}:{$variantId}";
    }

    /**
     * Nettoyer les réservations expirées (job de maintenance)
     * Les réservations Redis expirent automatiquement, cette méthode est un fallback
     *
     * @return int Nombre de réservations nettoyées
     */
    public function cleanupExpiredReservations(): int
    {
        $pattern = "stock:reservation:*";
        $keys = Redis::keys($pattern);
        $cleaned = 0;

        foreach ($keys as $key) {
            $ttl = Redis::ttl($key);
            if ($ttl === -2 || $ttl === -1) {
                // -2: clé n'existe plus, -1: pas d'expiration
                Redis::del($key);
                $cleaned++;
            }
        }

        return $cleaned;
    }
}
