<?php

namespace App\Services;

use App\Models\Order;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\Setting;
use App\Mail\NewOrderNotification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Exceptions\InsufficientStockException;

class OrderService
{
    protected StockService $stockService;

    public function __construct(StockService $stockService)
    {
        $this->stockService = $stockService;
    }

    /**
     * Créer une commande (avec ou sans compte)
     *
     * @param array $orderData Données complètes incluant les items
     * @return Order
     * @throws InsufficientStockException
     */
    public function createOrder(array $orderData): Order
    {
        // Extraire les items des données
        $items = $orderData['items'] ?? [];

        // Préparer les items avec informations produits
        $preparedItems = $this->prepareOrderItems($items);

        // ========================================
        // CRÉATION DE LA COMMANDE (TRANSACTION)
        // Le stock n'est plus un verrou : la source de prélèvement
        // est choisie par l'admin au moment du traitement/livraison.
        // ========================================

        return DB::transaction(function () use ($orderData, $preparedItems) {
            // 2.1 Calculer les montants
            $subtotal = collect($preparedItems)->sum(fn($item) => $item['subtotal']);
            $taxAmount = $orderData['tax_amount'] ?? 0;
            $shippingCost = $orderData['shipping_cost'] ?? 0;
            $totalAmount = $subtotal + $taxAmount + $shippingCost;

            // 2.2 Créer la commande
            $order = Order::create([
                'user_id' => $orderData['user_id'] ?? null,
                'customer_first_name' => $orderData['customer_first_name'],
                'customer_last_name' => $orderData['customer_last_name'],
                'customer_phone' => $orderData['customer_phone'],
                'customer_email' => $orderData['customer_email'] ?? null,
                'delivery_street' => $orderData['delivery_street'],
                'delivery_city' => $orderData['delivery_city'],
                'delivery_postal_code' => $orderData['delivery_postal_code'],
                'delivery_country' => $orderData['delivery_country'] ?? 'France',
                'store_id' => $orderData['store_id'] ?? null,
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'shipping_cost' => $shippingCost,
                'total_amount' => $totalAmount,
                'status' => 'EN_ATTENTE_TRAITEMENT',
                'payment_method' => 'CASH_ON_DELIVERY',
                'is_paid' => false,
                'customer_notes' => $orderData['customer_notes'] ?? null,
            ]);

            // Ajouter les items (snapshot des produits/prix)
            foreach ($preparedItems as $item) {
                $order->items()->create([
                    'product_id' => $item['product_id'],
                    'variant_id' => $item['variant_id'],
                    'product_name' => $item['product_name'],
                    'variant_size' => $item['variant_size'],
                    'unit_price' => $item['unit_price'],
                    'quantity' => $item['quantity'],
                    'subtotal' => $item['subtotal'],
                ]);
            }

            // Log status initial
            $order->statusHistory()->create([
                'old_status' => null,
                'new_status' => 'EN_ATTENTE_TRAITEMENT',
                'comment' => 'Commande créée',
                'changed_by' => $orderData['user_id'] ?? null,
            ]);

            // 2.5 Envoyer notification email aux admins
            $this->sendOrderNotification($order);

            // 2.6 Charger les relations pour la réponse
            return $order->load(['items', 'statusHistory']);
        });
    }

    /**
     * Envoyer notification email pour nouvelle commande
     */
    protected function sendOrderNotification(Order $order): void
    {
        // Vérifier si les notifications sont activées
        if (!Setting::get('order_notification_enabled', true)) {
            return;
        }

        // Récupérer les emails des admins
        $adminEmails = Setting::get('admin_notification_emails', []);

        if (empty($adminEmails)) {
            return;
        }

        // Dispatch en queue (non bloquant pour le checkout)
        try {
            foreach ($adminEmails as $email) {
                Mail::to($email)->queue(new NewOrderNotification($order));
            }
        } catch (\Exception $e) {
            // Log l'erreur mais ne pas faire échouer la commande
            logger()->error('Failed to dispatch order notification email', [
                'order_id' => $order->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Préparer les items de commande avec informations produits
     *
     * @param array $items
     * @return array
     */
    protected function prepareOrderItems(array $items): array
    {
        return collect($items)->map(function ($item) {
            // Récupérer la variante (qui contient product_id)
            $variant = ProductVariant::with('product')->findOrFail($item['variant_id']);

            // Récupérer le produit depuis la variante
            $product = $variant->product;

            if (!$product) {
                throw new \InvalidArgumentException(
                    "Le produit associé à la variante {$variant->id} est introuvable"
                );
            }

            $quantity = $item['quantity'];
            $unitPrice = $variant->price;
            $subtotal = $unitPrice * $quantity;

            return [
                'product_id' => $product->id,
                'variant_id' => $variant->id,
                'product_name' => $product->name,
                'variant_size' => $variant->size,
                'unit_price' => $unitPrice,
                'quantity' => $quantity,
                'subtotal' => $subtotal,
            ];
        })->toArray();
    }

    /**
     * Mettre à jour le statut d'une commande
     *
     * @param int $orderId
     * @param string $newStatus
     * @param string|null $comment
     * @param int|null $userId
     * @return Order
     */
    public function updateOrderStatus(
        int $orderId,
        string $newStatus,
        ?string $comment = null,
        ?int $userId = null
    ): Order {
        $order = Order::with(['items', 'store'])->findOrFail($orderId);

        // Mettre à jour le statut avec historique
        $order->updateStatus($newStatus, $comment, $userId);

        return $order->fresh(['items', 'store', 'statusHistory']);
    }

    /**
     * Annuler une commande et remettre le stock
     *
     * @param int $orderId
     * @param string|null $reason
     * @param int|null $userId
     * @return Order
     */
    public function cancelOrder(int $orderId, ?string $reason = null, ?int $userId = null): Order
    {
        $order = Order::with('items')->findOrFail($orderId);

        if (!$order->canBeCancelled()) {
            throw new \Exception("Cette commande ne peut plus être annulée (statut: {$order->status})");
        }

        // Mettre à jour le statut (pas de stock à libérer)
        $order->updateStatus('ANNULEE', $reason ?? 'Commande annulée', $userId);

        return $order->fresh(['items', 'store', 'statusHistory']);
    }

    /**
     * Marquer une commande comme livrée et confirmer la sortie du stock physique
     *
     * @param int $orderId
     * @param string|null $comment
     * @param int|null $userId
     * @return Order
     */
    public function deliverOrder(int $orderId, ?string $comment = null, ?int $userId = null, ?int $fulfillmentSourceId = null): Order
    {
        $order = Order::with('items')->findOrFail($orderId);

        if ($order->status === 'LIVREE') {
            throw new \Exception("Cette commande est déjà livrée");
        }

        if ($order->status === 'ANNULEE') {
            throw new \Exception("Une commande annulée ne peut pas être livrée");
        }

        return DB::transaction(function () use ($order, $comment, $userId, $fulfillmentSourceId) {
            // Si la source est de type STORE, tenter de décrémenter le stock
            if ($fulfillmentSourceId) {
                $source = \App\Models\FulfillmentSource::find($fulfillmentSourceId);
                if ($source && $source->type === 'STORE' && $order->store_id) {
                    foreach ($order->items as $item) {
                        $stockItem = \App\Models\StockItem::where('product_id', $item->product_id)
                            ->where('variant_id', $item->variant_id)
                            ->where('store_id', $order->store_id)
                            ->first();

                        if ($stockItem && $stockItem->quantity >= $item->quantity) {
                            $stockItem->quantity -= $item->quantity;
                            $stockItem->save();
                            $stockItem->logMovement('OUT', -$item->quantity, "Commande #{$order->order_number}", $userId);
                        }
                    }
                }
            }

            // Mettre à jour le statut
            $order->updateStatus('LIVREE', $comment ?? 'Commande livrée', $userId);

            return $order->fresh(['items', 'store', 'statusHistory', 'fulfillmentSource']);
        });
    }

    /**
     * Obtenir les commandes d'un utilisateur
     *
     * @param int $userId
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getUserOrders(int $userId)
    {
        return Order::with(['items.product', 'store'])
            ->where('user_id', $userId)
            ->orderByDesc('created_at')
            ->get();
    }

    /**
     * Obtenir une commande par numéro
     *
     * @param string $orderNumber
     * @param int|null $userId Si fourni, vérifie que la commande appartient à l'utilisateur
     * @return Order
     */
    public function getOrderByNumber(string $orderNumber, ?int $userId = null): Order
    {
        $query = Order::with(['items.product', 'items.variant', 'store'])
            ->where('order_number', $orderNumber);

        if ($userId !== null) {
            $query->where('user_id', $userId);
        }

        return $query->firstOrFail();
    }
}
