<?php

namespace App\Services;

use App\Models\ProductVariant;
use Illuminate\Support\Facades\Session;

class CartService
{
    private const SESSION_KEY = 'cart';

    /**
     * Get the current cart
     */
    public function getCart(): array
    {
        return Session::get(self::SESSION_KEY, [
            'items' => [],
            'total_items' => 0,
            'total_price' => 0,
        ]);
    }

    /**
     * Get cart items with full product data
     */
    public function getCartWithProducts(): array
    {
        $cart = $this->getCart();
        $items = [];

        foreach ($cart['items'] as $key => $item) {
            $variant = ProductVariant::with('product')->find($item['variant_id']);
            if ($variant && $variant->product) {
                $items[] = [
                    'key' => $key,
                    'variant_id' => $item['variant_id'],
                    'quantity' => $item['quantity'],
                    'variant' => $variant,
                    'product' => $variant->product,
                    'subtotal' => $variant->price * $item['quantity'],
                ];
            }
        }

        return [
            'items' => $items,
            'total_items' => $cart['total_items'],
            'total_price' => $cart['total_price'],
        ];
    }

    /**
     * Add an item to cart
     */
    public function addItem(int $variantId, int $quantity): array
    {
        $cart = $this->getCart();

        // Validate variant exists
        $variant = ProductVariant::with('product')->find($variantId);
        if (!$variant || !$variant->product) {
            throw new \InvalidArgumentException('Produit introuvable');
        }

        // Check if item already exists
        $key = $this->getItemKey($variantId);
        $existingQuantity = 0;

        if (isset($cart['items'][$key])) {
            $existingQuantity = $cart['items'][$key]['quantity'];
        }

        $newQuantity = $existingQuantity + $quantity;

        // Update or add item
        $cart['items'][$key] = [
            'variant_id' => $variantId,
            'quantity' => $newQuantity,
            'price' => $variant->price,
        ];

        // Recalculate totals
        $this->recalculateTotals($cart);

        Session::put(self::SESSION_KEY, $cart);

        return $cart;
    }

    /**
     * Update item quantity
     */
    public function updateQuantity(int $variantId, int $quantity): array
    {
        $cart = $this->getCart();
        $key = $this->getItemKey($variantId);

        if (!isset($cart['items'][$key])) {
            throw new \InvalidArgumentException('Article non trouvé dans le panier');
        }

        if ($quantity <= 0) {
            return $this->removeItem($variantId);
        }

        $variant = ProductVariant::find($variantId);
        $cart['items'][$key]['quantity'] = $quantity;
        $cart['items'][$key]['price'] = $variant->price;

        $this->recalculateTotals($cart);

        Session::put(self::SESSION_KEY, $cart);

        return $cart;
    }

    /**
     * Remove an item from cart
     */
    public function removeItem(int $variantId): array
    {
        $cart = $this->getCart();
        $key = $this->getItemKey($variantId);

        if (isset($cart['items'][$key])) {
            unset($cart['items'][$key]);
        }

        $this->recalculateTotals($cart);

        Session::put(self::SESSION_KEY, $cart);

        return $cart;
    }

    /**
     * Clear the cart
     */
    public function clearCart(): void
    {
        Session::forget(self::SESSION_KEY);
    }

    /**
     * Get item count
     */
    public function getItemCount(): int
    {
        $cart = $this->getCart();
        return $cart['total_items'];
    }

    /**
     * Check if cart is empty
     */
    public function isEmpty(): bool
    {
        $cart = $this->getCart();
        return empty($cart['items']);
    }

    /**
     * Prepare cart data for order creation
     */
    public function prepareForOrder(): array
    {
        $cart = $this->getCart();

        if (empty($cart['items'])) {
            throw new \InvalidArgumentException('Le panier est vide');
        }

        $items = [];
        foreach ($cart['items'] as $item) {
            $items[] = [
                'variant_id' => $item['variant_id'],
                'quantity' => $item['quantity'],
            ];
        }

        return [
            'items' => $items,
        ];
    }

    /**
     * Generate unique key for cart item
     */
    private function getItemKey(int $variantId): string
    {
        return 'variant_' . $variantId;
    }

    /**
     * Recalculate cart totals
     */
    private function recalculateTotals(array &$cart): void
    {
        $totalItems = 0;
        $totalPrice = 0;

        foreach ($cart['items'] as $item) {
            $totalItems += $item['quantity'];
            $totalPrice += $item['price'] * $item['quantity'];
        }

        $cart['total_items'] = $totalItems;
        $cart['total_price'] = $totalPrice;
    }
}
