<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Store extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'street',
        'city',
        'postal_code',
        'country',
        'phone',
        'email',
        'latitude',
        'longitude',
        'opening_hours',
        'is_active',
    ];

    protected $casts = [
        'latitude' => 'decimal:7',
        'longitude' => 'decimal:7',
        'opening_hours' => 'array',
        'is_active' => 'boolean',
    ];

    /**
     * Get stock items for this store
     */
    public function stockItems()
    {
        return $this->hasMany(StockItem::class);
    }

    /**
     * Get orders assigned to this store
     */
    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    /**
     * Scope active stores
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get full address attribute
     */
    public function getFullAddressAttribute(): string
    {
        return "{$this->street}, {$this->postal_code} {$this->city}, {$this->country}";
    }

    /**
     * Get stock for a specific product variant
     */
    public function getStockForVariant(int $productId, int $variantId): ?StockItem
    {
        return $this->stockItems()
            ->where('product_id', $productId)
            ->where('variant_id', $variantId)
            ->first();
    }

    /**
     * Check if store has sufficient stock for a variant
     */
    public function hasSufficientStock(int $productId, int $variantId, int $quantity = 1): bool
    {
        $stockItem = $this->getStockForVariant($productId, $variantId);
        return $stockItem && $stockItem->quantity >= $quantity;
    }

    /**
     * Get low stock items (below alert threshold)
     */
    public function getLowStockItems()
    {
        return $this->stockItems()
            ->whereColumn('quantity', '<=', 'alert_threshold')
            ->with(['product', 'variant'])
            ->get();
    }

    /**
     * Get total stock value in this store
     */
    public function getTotalStockValueAttribute(): float
    {
        return $this->stockItems()
            ->with('variant')
            ->get()
            ->sum(function ($stockItem) {
                return $stockItem->quantity * $stockItem->variant->price;
            });
    }
}
