<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class StockItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_id',
        'variant_id',
        'store_id',
        'quantity',
        'reserved_quantity',
        'alert_threshold',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'reserved_quantity' => 'integer',
        'alert_threshold' => 'integer',
    ];

    /**
     * Get the product
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the variant
     */
    public function variant()
    {
        return $this->belongsTo(ProductVariant::class, 'variant_id');
    }

    /**
     * Get the store
     */
    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    /**
     * Get stock movements
     */
    public function movements()
    {
        return $this->hasMany(StockMovement::class);
    }

    /**
     * Get available quantity (quantity - reserved_quantity)
     */
    public function getAvailableQuantityAttribute(): int
    {
        return max(0, $this->quantity - $this->reserved_quantity);
    }

    /**
     * Scope low stock items
     */
    public function scopeLowStock($query)
    {
        return $query->whereColumn('quantity', '<=', 'alert_threshold');
    }

    /**
     * Scope out of stock items
     */
    public function scopeOutOfStock($query)
    {
        return $query->where('quantity', 0);
    }

    /**
     * Check if stock is low
     */
    public function isLowStock(): bool
    {
        return $this->quantity <= $this->alert_threshold;
    }

    /**
     * Check if out of stock
     */
    public function isOutOfStock(): bool
    {
        return $this->quantity === 0;
    }

    /**
     * Check if sufficient stock available (prend en compte les réservations)
     */
    public function hasSufficientStock(int $requestedQuantity): bool
    {
        return $this->available_quantity >= $requestedQuantity;
    }

    /**
     * Check if sufficient physical stock (ignore les réservations)
     * Utilisé uniquement pour vérification administrative
     */
    public function hasSufficientPhysicalStock(int $requestedQuantity): bool
    {
        return $this->quantity >= $requestedQuantity;
    }

    /**
     * Log stock movement
     */
    public function logMovement(string $type, int $quantity, ?string $reason = null, ?int $performedBy = null): StockMovement
    {
        return $this->movements()->create([
            'type' => $type,
            'quantity' => $quantity,
            'reason' => $reason,
            'performed_by' => $performedBy,
        ]);
    }
}
