<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductVariant extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_id',
        'size',
        'price',
        'sku',
        'is_active',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    /**
     * Boot method pour générer automatiquement le SKU
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($variant) {
            if (empty($variant->sku)) {
                $variant->sku = $variant->generateSku();
            }
        });
    }

    /**
     * Generate unique SKU
     */
    protected function generateSku(): string
    {
        $productSlug = $this->product->slug ?? 'PROD';
        $size = str_replace('x', '-', $this->size);
        return strtoupper("{$productSlug}-{$size}-" . substr(md5(uniqid()), 0, 6));
    }

    /**
     * Get the product
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get stock items for this variant
     */
    public function stockItems()
    {
        return $this->hasMany(StockItem::class, 'variant_id');
    }

    /**
     * Get order items
     */
    public function orderItems()
    {
        return $this->hasMany(OrderItem::class, 'variant_id');
    }

    /**
     * Scope active variants
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get total stock for this variant across all stores
     */
    public function getTotalStockAttribute(): int
    {
        return $this->stockItems()->sum('quantity');
    }

    /**
     * Get stock in a specific store
     */
    public function getStockInStore(int $storeId): int
    {
        $stockItem = $this->stockItems()
            ->where('store_id', $storeId)
            ->first();

        return $stockItem ? $stockItem->quantity : 0;
    }

    /**
     * Check if variant is available in a specific store
     */
    public function isAvailableInStore(int $storeId, int $quantity = 1): bool
    {
        return $this->getStockInStore($storeId) >= $quantity;
    }
    /**
     * Accessor for availability used in views.
     */
    public function getIsAvailableAttribute(): bool
    {
        return $this->is_active && $this->stockItems()
            ->whereRaw('quantity - reserved_quantity > 0')
            ->exists();
    }
}
