<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_number',
        'user_id',
        'customer_first_name',
        'customer_last_name',
        'customer_phone',
        'customer_email',
        'delivery_street',
        'delivery_city',
        'delivery_postal_code',
        'delivery_country',
        'store_id',
        'subtotal',
        'tax_amount',
        'shipping_cost',
        'total_amount',
        'status',
        'payment_method',
        'is_paid',
        'customer_notes',
        'admin_notes',
        'delivered_at',
        'fulfillment_source_id',
        'fulfillment_note',
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_cost' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'is_paid' => 'boolean',
        'delivered_at' => 'datetime',
    ];

    /**
     * Boot method pour générer automatiquement le order_number et UUID
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            if (empty($order->order_number)) {
                $order->order_number = $order->generateOrderNumber();
            }
            // Générer UUID pour accès public sécurisé
            if (empty($order->uuid)) {
                $order->uuid = \Illuminate\Support\Str::uuid();
            }
        });
    }

    /**
     * Generate unique order number
     */
    protected function generateOrderNumber(): string
    {
        $year = date('Y');
        $lastOrder = static::whereYear('created_at', $year)
            ->orderBy('id', 'desc')
            ->first();

        $number = $lastOrder ? intval(substr($lastOrder->order_number, -5)) + 1 : 1;

        return "CMD-{$year}-" . str_pad($number, 5, '0', STR_PAD_LEFT);
    }

    /**
     * Get the user (if order was placed with account)
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the assigned store
     */
    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    /**
     * Get order items
     */
    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get status history
     */
    public function statusHistory()
    {
        return $this->hasMany(OrderStatusHistory::class);
    }

    /**
     * Get the fulfillment source
     */
    public function fulfillmentSource()
    {
        return $this->belongsTo(FulfillmentSource::class);
    }

    /**
     * Scope by status
     */
    public function scopeByStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope pending orders
     */
    public function scopePending($query)
    {
        return $query->where('status', 'PENDING');
    }

    /**
     * Scope recent orders
     */
    public function scopeRecent($query, int $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Get customer full name
     */
    public function getCustomerFullNameAttribute(): string
    {
        return trim("{$this->customer_first_name} {$this->customer_last_name}");
    }

    /**
     * Get delivery address
     */
    public function getDeliveryAddressAttribute(): string
    {
        return "{$this->delivery_street}, {$this->delivery_postal_code} {$this->delivery_city}, {$this->delivery_country}";
    }

    /**
     * Update order status with history logging
     */
    public function updateStatus(string $newStatus, ?string $comment = null, ?int $changedBy = null): bool
    {
        $oldStatus = $this->status;

        // Log status change
        $this->statusHistory()->create([
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'comment' => $comment,
            'changed_by' => $changedBy,
        ]);

        // Update status
        $this->status = $newStatus;

        // If delivered, set delivered_at
        if ($newStatus === 'LIVREE' && !$this->delivered_at) {
            $this->delivered_at = now();
        }

        return $this->save();
    }

    /**
     * Check if order can be cancelled
     */
    public function canBeCancelled(): bool
    {
        return in_array($this->status, ['EN_ATTENTE_TRAITEMENT', 'EN_ATTENTE_LIVRAISON']);
    }

    /**
     * Check if order is delivered
     */
    public function isDelivered(): bool
    {
        return $this->status === 'LIVREE';
    }

    /**
     * Check if order is cancelled
     */
    public function isCancelled(): bool
    {
        return $this->status === 'ANNULEE';
    }
}
