<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class ProductResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'slug' => $this->slug,
            'description' => $this->description,
            'short_desc' => $this->short_desc,
            'base_price' => (float) $this->base_price,

            // Caractéristiques matelas
            'type' => $this->type,
            'type_label' => $this->getTypeLabel(),
            'firmness' => $this->firmness,
            'firmness_label' => $this->getFirmnessLabel(),
            'height' => $this->height,
            'material' => $this->material,
            'warranty' => $this->warranty,

            // SEO
            'meta_title' => $this->meta_title,
            'meta_desc' => $this->meta_desc,

            // Médias
            'images' => $this->images ?? [],
            'thumbnail' => $this->thumbnail,

            // Statut
            'is_active' => (bool) $this->is_active,
            'is_featured' => (bool) $this->is_featured,

            // Catégorie
            'category' => $this->when($this->relationLoaded('category'), function () {
                return $this->category ? [
                    'id' => $this->category->id,
                    'name' => $this->category->name,
                    'slug' => $this->category->slug,
                ] : null;
            }),

            // Variantes (tailles)
            'variants' => ProductVariantResource::collection($this->whenLoaded('variants')),

            // Stock global (total tous magasins) - quantity MOINS reserved_quantity
            'total_stock' => $this->when(
                $this->relationLoaded('stockItems'),
                function () {
                    return $this->stockItems->sum(function ($item) {
                        return $item->quantity - $item->reserved_quantity;
                    });
                }
            ),

            // Disponibilité - produit disponible SI stock disponible > 0
            'is_available' => $this->when(
                $this->relationLoaded('stockItems'),
                function () {
                    $available = $this->stockItems->sum(function ($item) {
                        return $item->quantity - $item->reserved_quantity;
                    });
                    return $available > 0;
                }
            ),

            // Dates
            'created_at' => $this->created_at?->toIso8601String(),
            'updated_at' => $this->updated_at?->toIso8601String(),
        ];
    }

    /**
     * Get type label in French
     */
    protected function getTypeLabel(): ?string
    {
        return match($this->type) {
            'FOAM' => 'Mousse',
            'SPRING' => 'Ressorts',
            'MEMORY_FOAM' => 'Mémoire de forme',
            'LATEX' => 'Latex',
            'HYBRID' => 'Hybride',
            default => null,
        };
    }

    /**
     * Get firmness label in French
     */
    protected function getFirmnessLabel(): ?string
    {
        return match($this->firmness) {
            'SOFT' => 'Souple',
            'MEDIUM' => 'Médium',
            'FIRM' => 'Ferme',
            'EXTRA_FIRM' => 'Très ferme',
            default => null,
        };
    }
}
