<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ProductController extends Controller
{
    /**
     * Display products listing
     */
    public function index(Request $request): View
    {
        $query = Product::with(['variants', 'category'])
            ->active();

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by firmness
        if ($request->filled('firmness')) {
            $query->where('firmness', $request->firmness);
        }

        // Search by name
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Sort
        $sort = $request->get('sort', 'newest');
        switch ($sort) {
            case 'price_asc':
                $query->orderBy('base_price', 'asc');
                break;
            case 'price_desc':
                $query->orderBy('base_price', 'desc');
                break;
            case 'name':
                $query->orderBy('name', 'asc');
                break;
            case 'newest':
            default:
                $query->orderBy('created_at', 'desc');
                break;
        }

        $products = $query->paginate(12)->withQueryString();
        $categories = Category::active()->ordered()->get();

        // Get unique types and firmness values for filters
        $types = Product::active()->distinct()->pluck('type')->filter();
        $firmnesses = Product::active()->distinct()->pluck('firmness')->filter();

        return view('products.index', [
            'products' => $products,
            'categories' => $categories,
            'types' => $types,
            'firmnesses' => $firmnesses,
            'currentFilters' => [
                'category' => $request->category,
                'type' => $request->type,
                'firmness' => $request->firmness,
                'search' => $request->search,
                'sort' => $sort,
            ],
        ]);
    }

    /**
     * Display single product
     */
    public function show(string $slug): View
    {
        $product = Product::with(['variants', 'category'])
            ->where('slug', $slug)
            ->active()
            ->firstOrFail();

        // Get related products
        $relatedProducts = Product::with(['variants'])
            ->active()
            ->where('id', '!=', $product->id)
            ->where(function ($query) use ($product) {
                $query->where('category_id', $product->category_id)
                    ->orWhere('type', $product->type);
            })
            ->take(4)
            ->get();

        return view('products.show', [
            'product' => $product,
            'relatedProducts' => $relatedProducts,
        ]);
    }
}
