<?php

namespace App\Http\Controllers;

use App\Services\CartService;
use App\Services\OrderService;
use App\Exceptions\InsufficientStockException;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class CheckoutController extends Controller
{
    public function __construct(
        private CartService $cartService,
        private OrderService $orderService
    ) {}

    /**
     * Display checkout page
     */
    public function index(): View|RedirectResponse
    {
        if ($this->cartService->isEmpty()) {
            return redirect()->route('cart.index')
                ->with('error', 'Votre panier est vide');
        }

        $cart = $this->cartService->getCartWithProducts();

        return view('checkout.index', [
            'cart' => $cart,
        ]);
    }

    /**
     * Process order
     */
    public function store(Request $request): RedirectResponse
    {
        if ($this->cartService->isEmpty()) {
            return redirect()->route('cart.index')
                ->with('error', 'Votre panier est vide');
        }

        $validated = $request->validate([
            'customer_first_name' => 'required|string|max:100',
            'customer_last_name' => 'required|string|max:100',
            'customer_phone' => 'required|string|max:20',
            'customer_email' => 'nullable|email|max:255',
            'delivery_street' => 'required|string|max:255',
            'delivery_city' => 'required|string|max:100',
            'delivery_postal_code' => 'nullable|string|max:20',
            'notes' => 'nullable|string|max:1000',
        ]);

        try {
            $cartData = $this->cartService->prepareForOrder();

            $orderData = array_merge($validated, [
                'items' => $cartData['items'],
                'user_id' => auth()->id(), // null si guest
                'customer_notes' => $validated['notes'] ?? null,
            ]);

            $order = $this->orderService->createOrder($orderData);

            // Clear cart after successful order
            $this->cartService->clearCart();

            return redirect()->route('checkout.confirmation', $order->uuid)
                ->with('success', 'Votre commande a été enregistrée avec succès !');

        } catch (InsufficientStockException $e) {
            return redirect()->route('checkout.index')
                ->with('error', 'Stock insuffisant : ' . $e->getMessage());
        } catch (\Exception $e) {
            return redirect()->route('checkout.index')
                ->with('error', 'Une erreur est survenue : ' . $e->getMessage());
        }
    }

    /**
     * Display order confirmation (accessed via non-guessable UUID)
     */
    public function confirmation(string $uuid): View
    {
        $order = \App\Models\Order::with(['items.product', 'items.variant'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        return view('checkout.confirmation', [
            'order' => $order,
        ]);
    }
}
