<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Mail\AccountActivatedMail;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * List all users with optional role filter
     */
    public function index(Request $request)
    {
        $query = User::query();

        // Filter by role
        if ($request->has('role') && in_array($request->role, ['ADMIN', 'CUSTOMER'])) {
            $query->where('role', $request->role);
        }

        // Search by name or email
        if ($request->has('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('first_name', 'like', '%' . $request->search . '%')
                  ->orWhere('last_name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        // Filter by active status
        if ($request->has('is_active')) {
            $query->where('is_active', filter_var($request->is_active, FILTER_VALIDATE_BOOLEAN));
        }

        $users = $query->orderBy('created_at', 'desc')->paginate(50);

        return response()->json($users);
    }

    /**
     * Get a specific user with their orders
     */
    public function show($id)
    {
        $user = User::with('orders')->findOrFail($id);
        return response()->json($user);
    }

    /**
     * Create a new user (admin only)
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'role' => 'required|in:ADMIN,CUSTOMER',
            'is_active' => 'boolean',
        ]);

        $user = User::create([
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'phone' => $validated['phone'] ?? null,
            'role' => $validated['role'],
            'is_active' => $validated['is_active'] ?? true,
        ]);

        return response()->json([
            'message' => 'User created successfully',
            'user' => $user,
        ], 201);
    }

    /**
     * Update user details
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $validated = $request->validate([
            'email' => ['sometimes', 'email', Rule::unique('users')->ignore($user->id)],
            'password' => 'sometimes|string|min:8',
            'first_name' => 'sometimes|string|max:255',
            'last_name' => 'sometimes|string|max:255',
            'phone' => 'nullable|string|max:20',
            'role' => 'sometimes|in:ADMIN,CUSTOMER',
            'is_active' => 'sometimes|boolean',
        ]);

        // Hash password if provided
        if (isset($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        }

        $user->update($validated);

        return response()->json([
            'message' => 'User updated successfully',
            'user' => $user->fresh(),
        ]);
    }

    /**
     * Toggle user active status
     */
    public function toggleActive($id)
    {
        $user = User::findOrFail($id);

        // Prevent deactivating yourself
        if ($user->id === auth()->id()) {
            return response()->json([
                'message' => 'Vous ne pouvez pas désactiver votre propre compte',
            ], 403);
        }

        $wasInactive = !$user->is_active;
        $user->is_active = !$user->is_active;
        $user->save();

        // Send email when activating an account
        if ($wasInactive && $user->is_active && $user->email_verified_at) {
            try {
                Mail::to($user->email)->send(new AccountActivatedMail($user));
            } catch (\Exception $e) {
                logger()->error('Failed to send account activation email', [
                    'user_id' => $user->id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        return response()->json([
            'message' => 'Statut utilisateur mis à jour avec succès',
            'user' => $user,
        ]);
    }

    /**
     * Activate a user account (admin only)
     */
    public function activate($id)
    {
        $user = User::findOrFail($id);

        if ($user->is_active) {
            return response()->json([
                'message' => 'Le compte est déjà actif',
            ], 400);
        }

        if (!$user->email_verified_at) {
            return response()->json([
                'message' => 'L\'email doit être vérifié avant l\'activation',
            ], 400);
        }

        $user->is_active = true;
        $user->save();

        // Send activation email
        try {
            Mail::to($user->email)->send(new AccountActivatedMail($user));
        } catch (\Exception $e) {
            logger()->error('Failed to send account activation email', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
        }

        return response()->json([
            'message' => 'Compte activé avec succès et email envoyé au client',
            'user' => $user,
        ]);
    }

    /**
     * Delete a user
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);

        // Prevent deleting yourself
        if ($user->id === auth()->id()) {
            return response()->json([
                'message' => 'You cannot delete your own account',
            ], 403);
        }

        // Check if user has orders
        if ($user->orders()->count() > 0) {
            return response()->json([
                'message' => 'Cannot delete user with existing orders. Consider deactivating instead.',
            ], 400);
        }

        $user->delete();

        return response()->json([
            'message' => 'User deleted successfully',
        ]);
    }
}
