<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ProductResource;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\StockItem;
use App\Models\Store;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        // Charger les relations nécessaires incluant stockItems pour calculer is_available
        $query = Product::with(['category', 'variants', 'stockItems']);

        // Par défaut, on filtre les produits actifs pour l'API publique
        // Sauf si le paramètre include_inactive=true est passé (réservé admin)
        if (!$request->has('include_inactive') || !$request->boolean('include_inactive')) {
            $query->where('is_active', true);
        }

        // Search by name
        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Filter by category
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        $perPage = $request->get('per_page', 12);
        $products = $query->paginate($perPage);

        // Utiliser ProductResource pour exposer is_available et total_stock
        return ProductResource::collection($products);
    }

    public function show($id)
    {
        // Charger stockItems au niveau produit pour is_available + au niveau variant pour détails
        // Accepter slug OU id (si c'est un nombre, cherche par ID, sinon par slug)
        $product = Product::with(['category', 'variants', 'stockItems', 'variants.stockItems.store'])
            ->where(is_numeric($id) ? 'id' : 'slug', $id)
            ->where('is_active', true)  // Seulement les produits actifs pour l'API publique
            ->firstOrFail();

        // Utiliser ProductResource pour exposer is_available et total_stock
        return response()->json([
            'data' => new ProductResource($product),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:products',
            'description' => 'required|string',
            'base_price' => 'required|numeric|min:0',
            'type' => 'nullable|in:FOAM,SPRING,MEMORY_FOAM,LATEX,HYBRID',
            'firmness' => 'nullable|in:SOFT,MEDIUM,FIRM,EXTRA_FIRM',
            'category_id' => 'required|exists:categories,id',
            'images' => 'nullable|array',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'variants' => 'required|array|min:1',
            'variants.*.size' => 'required|string',
            'variants.*.price' => 'required|numeric|min:0',
            'initial_stock' => 'required|integer|min:0',
        ]);

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        // Images will be automatically handled by Laravel's array cast
        // No need to json_encode - the model cast does it

        // Create product
        $product = Product::create($validated);

        // Create variants and stock
        $stores = Store::all();
        foreach ($validated['variants'] as $variantData) {
            $variant = ProductVariant::create([
                'product_id' => $product->id,
                'size' => $variantData['size'],
                'price' => $variantData['price'],
            ]);

            // Create stock for each store
            foreach ($stores as $store) {
                StockItem::create([
                    'product_id' => $product->id,
                    'variant_id' => $variant->id,
                    'store_id' => $store->id,
                    'quantity' => $validated['initial_stock'],
                    'reserved_quantity' => 0,
                    'alert_threshold' => 5,
                ]);
            }
        }

        // Vérifier que le stock a bien été créé
        $stockCreated = StockItem::where('product_id', $product->id)->count();
        $expectedStock = count($validated['variants']) * $stores->count();

        if ($stockCreated !== $expectedStock) {
            \Log::error('Stock creation failed', [
                'product_id' => $product->id,
                'product_name' => $product->name,
                'expected' => $expectedStock,
                'created' => $stockCreated,
            ]);

            // Annuler la création du produit
            $product->delete();

            return response()->json([
                'message' => 'Erreur lors de la création du stock. Produit annulé.',
                'error_type' => 'stock_creation_failed',
                'expected' => $expectedStock,
                'created' => $stockCreated,
            ], 500);
        }

        return response()->json($product->load(['category', 'variants']), 201);
    }

    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'slug' => 'sometimes|nullable|string|max:255|unique:products,slug,' . $id,
            'description' => 'sometimes|required|string',
            'base_price' => 'sometimes|required|numeric|min:0',
            'type' => 'nullable|in:FOAM,SPRING,MEMORY_FOAM,LATEX,HYBRID',
            'firmness' => 'nullable|in:SOFT,MEDIUM,FIRM,EXTRA_FIRM',
            'category_id' => 'sometimes|required|exists:categories,id',
            'images' => 'nullable|array',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
        ]);

        // Images will be automatically handled by Laravel's array cast
        // No need to json_encode - the model cast does it

        $product->update($validated);

        return response()->json($product->load(['category', 'variants']));
    }

    public function destroy($id)
    {
        $product = Product::findOrFail($id);

        // Check if product has orders
        $hasOrders = DB::table('order_items')->where('product_id', $id)->exists();

        if ($hasOrders) {
            return response()->json([
                'error_type' => 'product_has_orders',
                'message' => 'Ce produit ne peut pas être supprimé car il est lié à des commandes existantes. Vous pouvez le désactiver pour qu\'il ne soit plus visible sur le site.',
                'can_deactivate' => true,
            ], 400);
        }

        // Delete associated stock items
        StockItem::where('product_id', $id)->delete();

        // Delete variants
        ProductVariant::where('product_id', $id)->delete();

        // Delete product
        $product->delete();

        return response()->json([
            'message' => 'Product deleted successfully'
        ]);
    }

    /**
     * Toggle product active status
     */
    public function toggleActive($id)
    {
        $product = Product::findOrFail($id);
        $product->is_active = !$product->is_active;
        $product->save();

        return response()->json([
            'message' => $product->is_active ? 'Produit activé avec succès' : 'Produit désactivé avec succès',
            'is_active' => $product->is_active,
            'data' => $product->load(['category', 'variants']),
        ]);
    }

    /**
     * Upload product images
     */
    public function uploadImages(Request $request)
    {
        $request->validate([
            'images' => 'required|array|max:5',
            'images.*' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048', // 2MB max
        ]);

        $uploadedPaths = [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                // Generate unique filename
                $filename = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();

                // Store in public/storage/products
                $path = $image->storeAs('products', $filename, 'public');

                // Return full URL
                $uploadedPaths[] = Storage::url($path);
            }
        }

        return response()->json([
            'message' => 'Images uploaded successfully',
            'images' => $uploadedPaths,
        ]);
    }

    /**
     * Delete a product image
     */
    public function deleteImage(Request $request)
    {
        $request->validate([
            'image_path' => 'required|string',
        ]);

        $imagePath = $request->image_path;

        // Extract the storage path from the URL
        // URL format: /storage/products/filename.jpg
        $storagePath = str_replace('/storage/', '', $imagePath);

        if (Storage::disk('public')->exists($storagePath)) {
            Storage::disk('public')->delete($storagePath);

            return response()->json([
                'message' => 'Image deleted successfully',
            ]);
        }

        return response()->json([
            'message' => 'Image not found',
        ], 404);
    }
}
