<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Store;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class StoreController extends Controller
{
    public function index(Request $request): View
    {
        $query = Store::query();

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('city', 'like', '%' . $request->search . '%');
            });
        }

        if ($request->filled('active')) {
            $query->where('is_active', $request->active === 'true');
        }

        $stores = $query->orderBy('name')->paginate(20)->withQueryString();

        return view('admin.stores.index', [
            'stores' => $stores,
            'currentFilters' => [
                'search' => $request->search,
                'active' => $request->active,
            ],
        ]);
    }

    public function create(): View
    {
        return view('admin.stores.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'street' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'opening_hours' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['country'] = $request->filled('country') ? $validated['country'] : 'France';
        $validated['postal_code'] = $request->filled('postal_code') ? $validated['postal_code'] : null;

        $openingHours = $this->parseOpeningHours($request->input('opening_hours'));
        if ($request->filled('opening_hours') && $openingHours === null) {
            return redirect()->back()
                ->withErrors(['opening_hours' => 'Format des horaires invalide.'])
                ->withInput();
        }
        $validated['opening_hours'] = $openingHours;

        Store::create($validated);

        return redirect()->route('admin.stores.index')
            ->with('success', 'Magasin cree avec succes');
    }

    public function edit(int $id): View
    {
        $store = Store::findOrFail($id);

        return view('admin.stores.edit', [
            'store' => $store,
        ]);
    }

    public function update(Request $request, int $id): RedirectResponse
    {
        $store = Store::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'street' => 'required|string|max:255',
            'city' => 'required|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'phone' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'opening_hours' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        if (!$request->filled('country')) {
            unset($validated['country']);
        }
        $validated['postal_code'] = $request->filled('postal_code') ? $validated['postal_code'] : null;

        $openingHours = $this->parseOpeningHours($request->input('opening_hours'));
        if ($request->filled('opening_hours') && $openingHours === null) {
            return redirect()->back()
                ->withErrors(['opening_hours' => 'Format des horaires invalide.'])
                ->withInput();
        }
        $validated['opening_hours'] = $openingHours;

        $store->update($validated);

        return redirect()->route('admin.stores.edit', $id)
            ->with('success', 'Magasin mis a jour avec succes');
    }

    public function destroy(int $id): RedirectResponse
    {
        $store = Store::findOrFail($id);
        $store->delete();

        return redirect()->route('admin.stores.index')
            ->with('success', 'Magasin supprime avec succes');
    }

    public function toggleActive(int $id): RedirectResponse
    {
        $store = Store::findOrFail($id);
        $store->is_active = !$store->is_active;
        $store->save();

        return redirect()->back()
            ->with('success', 'Statut du magasin mis a jour');
    }

    private function parseOpeningHours(?string $input): ?array
    {
        if ($input === null) {
            return null;
        }

        $trimmed = trim($input);
        if ($trimmed === '') {
            return null;
        }

        $decoded = json_decode($trimmed, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return $decoded;
        }

        $lines = preg_split('/\r\n|\r|\n/', $trimmed);
        $result = [];

        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }

            if (strpos($line, ':') !== false) {
                [$day, $hours] = array_map('trim', explode(':', $line, 2));
            } elseif (strpos($line, '=') !== false) {
                [$day, $hours] = array_map('trim', explode('=', $line, 2));
            } else {
                return null;
            }

            if ($day === '' || $hours === '') {
                return null;
            }

            $result[$day] = $hours;
        }

        return empty($result) ? null : $result;
    }
}
