<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\FulfillmentSource;
use App\Services\OrderService;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class OrderController extends Controller
{
    public function __construct(
        private OrderService $orderService
    ) {}

    public function index(Request $request): View
    {
        $query = Order::with(['store', 'items.product', 'items.variant']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by store
        if ($request->filled('store_id')) {
            $query->where('store_id', $request->store_id);
        }

        // Search by order number or customer name
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                    ->orWhere('customer_first_name', 'like', "%{$search}%")
                    ->orWhere('customer_last_name', 'like', "%{$search}%")
                    ->orWhere('customer_phone', 'like', "%{$search}%");
            });
        }

        $orders = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();

        $stores = \App\Models\Store::active()->get();

        $statuses = [
            'EN_ATTENTE_TRAITEMENT' => 'En attente',
            'EN_ATTENTE_LIVRAISON' => 'En livraison',
            'LIVREE' => 'Livrée',
            'ANNULEE' => 'Annulée',
        ];

        return view('admin.orders.index', [
            'orders' => $orders,
            'stores' => $stores,
            'statuses' => $statuses,
            'currentFilters' => [
                'status' => $request->status,
                'store_id' => $request->store_id,
                'search' => $request->search,
            ],
        ]);
    }

    public function show(int $id): View
    {
        $order = Order::with(['store', 'items.product', 'items.variant', 'statusHistory.changedBy', 'fulfillmentSource'])
            ->findOrFail($id);

        $statuses = [
            'EN_ATTENTE_TRAITEMENT' => 'En attente',
            'EN_ATTENTE_LIVRAISON' => 'En livraison',
            'LIVREE' => 'Livrée',
            'ANNULEE' => 'Annulée',
        ];

        $fulfillmentSources = FulfillmentSource::active()->orderBy('name')->get();

        return view('admin.orders.show', [
            'order' => $order,
            'statuses' => $statuses,
            'fulfillmentSources' => $fulfillmentSources,
        ]);
    }

    public function updateStatus(Request $request, int $id): RedirectResponse
    {
        $rules = [
            'status' => 'required|in:EN_ATTENTE_TRAITEMENT,EN_ATTENTE_LIVRAISON,LIVREE,ANNULEE',
            'notes' => 'nullable|string|max:500',
            'fulfillment_source_id' => 'nullable|exists:fulfillment_sources,id',
            'fulfillment_note' => 'nullable|string|max:500',
        ];

        // Source obligatoire pour EN_ATTENTE_LIVRAISON et LIVREE
        if (in_array($request->status, ['EN_ATTENTE_LIVRAISON', 'LIVREE'])) {
            $rules['fulfillment_source_id'] = 'required|exists:fulfillment_sources,id';
        }

        $request->validate($rules, [
            'fulfillment_source_id.required' => 'La source de prélèvement est obligatoire pour ce statut.',
        ]);

        try {
            $order = Order::findOrFail($id);
            $newStatus = $request->status;
            $notes = $request->notes ?? '';
            $userId = auth()->id();

            // Sauvegarder la source de prélèvement si fournie
            if ($request->filled('fulfillment_source_id')) {
                $order->fulfillment_source_id = $request->fulfillment_source_id;
                $order->fulfillment_note = $request->fulfillment_note;
                $order->save();
            }

            if ($newStatus === 'LIVREE') {
                $this->orderService->deliverOrder($id, $notes, $userId, $request->fulfillment_source_id);
            } elseif ($newStatus === 'ANNULEE') {
                $this->orderService->cancelOrder($id, $notes, $userId);
            } else {
                $order->updateStatus($newStatus, $notes, $userId);
            }

            return redirect()->route('admin.orders.show', $id)
                ->with('success', 'Statut mis à jour avec succès');
        } catch (\Exception $e) {
            return redirect()->route('admin.orders.show', $id)
                ->with('error', 'Erreur: ' . $e->getMessage());
        }
    }
}
